<?php
// app/Cron.php - Minimal 5-field cron parser (min hour dom mon dow)
// Supports: "*", "*/n", "a-b", "a,b,c", ranges with steps "a-b/n"
// DOW: 0 or 7 = Sunday.
declare(strict_types=1);

namespace App;

final class Cron {
  private string $expr;
  private array $fields; // [min, hour, dom, mon, dow]

  public function __construct(string $expr) {
    $this->expr = trim($expr);
    $parts = preg_split('/\s+/', $this->expr);
    if (!$parts || count($parts) !== 5) {
      throw new \InvalidArgumentException("Cron must have 5 fields: '* * * * *'");
    }
    $this->fields = $parts;
  }

  public function isDue(\DateTimeInterface $dt): bool {
    $m  = (int)$dt->format('i');          // 0-59
    $h  = (int)$dt->format('G');          // 0-23
    $dom= (int)$dt->format('j');          // 1-31
    $mon= (int)$dt->format('n');          // 1-12
    $dow= (int)$dt->format('w');          // 0-6 (Sun=0)
    return (
      $this->matchField($this->fields[0], $m, 0, 59) &&
      $this->matchField($this->fields[1], $h, 0, 23) &&
      $this->matchField($this->fields[2], $dom, 1, 31) &&
      $this->matchField($this->fields[3], $mon, 1, 12) &&
      $this->matchFieldDow($this->fields[4], $dow)
    );
  }

  public function getNextRunDate(\DateTimeInterface $from): \DateTimeImmutable {
    // Brute-force search up to 365 days ahead, checking each minute.
    $ts = $from->getTimestamp();
    $start = $ts - ($ts % 60) + 60; // next minute boundary
    for ($i = 0; $i < 365*24*60; $i++) {
      $t = $start + 60*$i;
      $dt = (new \DateTimeImmutable())->setTimestamp($t);
      if ($this->isDue($dt)) return $dt;
    }
    // Fallback: return +5 minutes
    return (new \DateTimeImmutable('@'.($start+60*5)))->setTimezone($from->getTimezone());
  }

  private function matchFieldDow(string $expr, int $val): bool {
    // Accept 0 or 7 as Sunday
    $fn = function(int $x): int { return $x == 7 ? 0 : $x; };
    // translate ranges & lists accordingly
    if ($expr == '*') return true;
    foreach (explode(',', $expr) as $seg) {
      if ($this->matchSegment($seg, $fn($val), 0, 6, true)) return true;
    }
    return false;
  }

  private function matchField(string $expr, int $val, int $min, int $max): bool {
    if ($expr == '*') return true;
    foreach (explode(',', $expr) as $seg) {
      if ($this->matchSegment($seg, $val, $min, $max, false)) return true;
    }
    return false;
  }

  private function matchSegment(string $seg, int $val, int $min, int $max, bool $dowMode): bool {
    $seg = trim($seg);
    if ($seg === '*') return true;
    $step = 1;
    if (strpos($seg, '/') !== false) {
      [$base, $s] = explode('/', $seg, 2);
      $step = max(1, (int)$s);
    } else {
      $base = $seg;
    }
    if ($base === '*') {
      return (($val - $min) % $step) == 0;
    }
    if (strpos($base, '-') !== false) {
      [$a, $b] = explode('-', $base, 2);
      $a = (int)$a; $b = (int)$b;
      if ($dowMode) { if ($a==7) $a=0; if ($b==7) $b=0; }
      if ($a <= $b) {
        if ($val < $a || $val > $b) return false;
        return (($val - $a) % $step) == 0;
      } else {
        // wrap-around range like 5-2
        $in = ($val >= $a || $val <= $b);
        if (!$in) return false;
        // treat start as anchor for step
        $anchor = $val >= $a ? $a : ($min); // not perfect but acceptable
        return (($val - $anchor) % $step) == 0;
      }
    }
    $n = (int)$base;
    if ($dowMode && $n==7) $n=0;
    return $val == $n;
  }
}
