<?php
// app/JobRepository.php
declare(strict_types=1);

namespace App;

use PDO;

final class JobRepository {
  public function __construct(private PDO $pdo) {}

  public function all(): array {
    return $this->pdo->query("SELECT * FROM jobs ORDER BY id DESC")->fetchAll();
  }

  public function find(int $id): ?array {
    $st = $this->pdo->prepare("SELECT * FROM jobs WHERE id=?");
    $st->execute([$id]);
    $r = $st->fetch();
    return $r ?: null;
  }

  public function create(array $d): int {
    $st = $this->pdo->prepare("
      INSERT INTO jobs(name,url,method,headers,body,cron_expr,enabled,timeout_sec,proxy_url,proxy_auth,verify_ssl,created_at,updated_at)
      VALUES(:name,:url,:method,:headers,:body,:cron_expr,:enabled,:timeout_sec,:proxy_url,:proxy_auth,:verify_ssl,datetime('now'),datetime('now'))
    ");
    $st->execute([
      ':name'=>$d['name'], ':url'=>$d['url'], ':method'=>$d['method'] ?? 'GET',
      ':headers'=>$d['headers'] ?? null, ':body'=>$d['body'] ?? null,
      ':cron_expr'=>$d['cron_expr'], ':enabled'=>($d['enabled'] ?? 1),
      ':timeout_sec'=>($d['timeout_sec'] ?? 30),
      ':proxy_url'=>$d['proxy_url'] ?? null, ':proxy_auth'=>$d['proxy_auth'] ?? null,
      ':verify_ssl'=>$d['verify_ssl'] ?? 1,
    ]);
    return (int)$this->pdo->lastInsertId();
  }

  public function update(int $id, array $d): void {
    $st = $this->pdo->prepare("
      UPDATE jobs SET name=:name,url=:url,method=:method,headers=:headers,body=:body,
      cron_expr=:cron_expr,enabled=:enabled,timeout_sec=:timeout_sec,proxy_url=:proxy_url,proxy_auth=:proxy_auth,verify_ssl=:verify_ssl,
      updated_at=datetime('now') WHERE id=:id
    ");
    $st->execute([
      ':id'=>$id, ':name'=>$d['name'], ':url'=>$d['url'], ':method'=>$d['method'] ?? 'GET',
      ':headers'=>$d['headers'] ?? null, ':body'=>$d['body'] ?? null,
      ':cron_expr'=>$d['cron_expr'], ':enabled'=>($d['enabled'] ?? 1),
      ':timeout_sec'=>($d['timeout_sec'] ?? 30),
      ':proxy_url'=>$d['proxy_url'] ?? null, ':proxy_auth'=>$d['proxy_auth'] ?? null,
      ':verify_ssl'=>$d['verify_ssl'] ?? 1,
    ]);
  }

  public function delete(int $id): void {
    $st = $this->pdo->prepare("DELETE FROM jobs WHERE id=?");
    $st->execute([$id]);
  }

  public function logs(int $jobId, int $limit = 50): array {
    $st = $this->pdo->prepare("SELECT * FROM job_logs WHERE job_id=? ORDER BY id DESC LIMIT ?");
    $st->execute([$jobId, $limit]);
    return $st->fetchAll();
  }

  public function insertLog(array $d): void {
    $st = $this->pdo->prepare("
      INSERT INTO job_logs(job_id,started_at,finished_at,success,http_code,duration_ms,response_preview,error)
      VALUES(:job_id,:started_at,:finished_at,:success,:http_code,:duration_ms,:response_preview,:error)
    ");
    $st->execute($d);
  }

  public function touchLastRun(int $id, string $status, ?int $code, ?string $err): void {
    $st = $this->pdo->prepare("
      UPDATE jobs SET last_run_at=datetime('now'), last_status=:status, last_http_code=:code, last_error=:err
      WHERE id=:id
    ");
    $st->execute([':status'=>$status, ':code'=>$code, ':err'=>$err, ':id'=>$id]);
  }
}
