#!/usr/bin/env php
<?php
declare(strict_types=1);

require __DIR__ . '/../app/db.php';
require __DIR__ . '/../app/helpers.php';
require __DIR__ . '/../app/Cron.php';
require __DIR__ . '/../app/JobRepository.php';

use App\JobRepository;
use App\Cron;

$config = require __DIR__ . '/../app/config.php';
$repo = new JobRepository($pdo);

// Get enabled jobs
$jobs = $pdo->query("SELECT * FROM jobs WHERE enabled=1")->fetchAll();
$now = new DateTimeImmutable('now');

foreach ($jobs as $j) {
  try {
    $cron = new Cron($j['cron_expr']);
    if (!$cron->isDue($now)) continue;
  } catch (Throwable $e) {
    // Invalid cron: skip and mark error
    $repo->touchLastRun((int)$j['id'], 'FAIL', null, 'Invalid cron: '.$e->getMessage());
    continue;
  }

  $ch = curl_init();
  apply_proxy_and_ssl($j, $ch, $config);

  $headers = json_or_null($j['headers']) ?? [];
  $hdrs = [];
  foreach ($headers as $k=>$v) $hdrs[] = $k.': '.$v;
  $method = strtoupper($j['method'] ?: 'GET');
  $opts = [
    CURLOPT_URL => $j['url'],
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => (int)$j['timeout_sec'],
    CURLOPT_HEADER => false,
    CURLOPT_HTTPHEADER => $hdrs
  ];
  if ($method !== 'GET') {
    $opts[CURLOPT_CUSTOMREQUEST] = $method;
    if (!empty($j['body'])) $opts[CURLOPT_POSTFIELDS] = $j['body'];
  }
  curl_setopt_array($ch, $opts);
  $start = microtime(true);
  $resp = curl_exec($ch);
  $err  = curl_error($ch) ?: null;
  $code = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
  curl_close($ch);
  $duration = (int)round((microtime(true)-$start)*1000);

  $ok = $err === null && $code >= 200 && $code < 400;
  $repo->insertLog([
    ':job_id'=>(int)$j['id'],
    ':started_at'=>gmdate('Y-m-d H:i:s'),
    ':finished_at'=>gmdate('Y-m-d H:i:s'),
    ':success'=>$ok ? 1 : 0,
    ':http_code'=>$code ?: null,
    ':duration_ms'=>$duration,
    ':response_preview'=>shorten($resp ?? '', 500),
    ':error'=>$err
  ]);
  $repo->touchLastRun((int)$j['id'], $ok ? 'OK' : 'FAIL', $code ?: null, $err);
}
