<?php
declare(strict_types=1);

// Optional basic auth
// $USER='admin'; $PASS='secret';
// if (!isset($_SERVER['PHP_AUTH_USER']) || $_SERVER['PHP_AUTH_USER']!==$USER || $_SERVER['PHP_AUTH_PW']!==$PASS) {
//   header('WWW-Authenticate: Basic realm="WebCron"'); header('HTTP/1.0 401 Unauthorized'); exit('Auth required');
// }

require __DIR__ . '/../app/db.php';
require __DIR__ . '/../app/helpers.php';
require __DIR__ . '/../app/Cron.php';
require __DIR__ . '/../app/JobRepository.php';

use App\JobRepository;
use App\Cron;

$config = require __DIR__ . '/../app/config.php';
$repo = new JobRepository($pdo);
$action = $_GET['a'] ?? 'list';

// Handle POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if ($action === 'create') {
    $repo->create([
      'name'=>$_POST['name'], 'url'=>$_POST['url'], 'method'=>$_POST['method'],
      'headers'=>$_POST['headers'] ?: null, 'body'=>$_POST['body'] ?: null,
      'cron_expr'=>$_POST['cron_expr'], 'enabled'=>isset($_POST['enabled'])?1:0,
      'timeout_sec'=>(int)($_POST['timeout_sec'] ?? ($config['default_timeout_sec'] ?? 30)),
      'proxy_url'=>$_POST['proxy_url'] ?: null,
      'proxy_auth'=>$_POST['proxy_auth'] ?: null,
      'verify_ssl'=>isset($_POST['verify_ssl']) ? 1 : 0,
    ]);
    header('Location: ?a=list'); exit;
  }
  if ($action === 'edit' && isset($_GET['id'])) {
    $repo->update((int)$_GET['id'], [
      'name'=>$_POST['name'], 'url'=>$_POST['url'], 'method'=>$_POST['method'],
      'headers'=>$_POST['headers'] ?: null, 'body'=>$_POST['body'] ?: null,
      'cron_expr'=>$_POST['cron_expr'], 'enabled'=>isset($_POST['enabled'])?1:0,
      'timeout_sec'=>(int)($_POST['timeout_sec'] ?? ($config['default_timeout_sec'] ?? 30)),
      'proxy_url'=>$_POST['proxy_url'] ?: null,
      'proxy_auth'=>$_POST['proxy_auth'] ?: null,
      'verify_ssl'=>isset($_POST['verify_ssl']) ? 1 : 0,
    ]);
    header('Location: ?a=list'); exit;
  }
}

if ($action === 'delete' && isset($_GET['id'])) {
  $repo->delete((int)$_GET['id']); header('Location: ?a=list'); exit;
}

function run_now(JobRepository $repo, array $job, array $config): array {
  $ch = curl_init();
  apply_proxy_and_ssl($job, $ch, $config);

  $headers = json_or_null($job['headers']) ?? [];
  $hdrs = [];
  foreach ($headers as $k=>$v) $hdrs[] = $k.': '.$v;

  $method = strtoupper($job['method'] ?: 'GET');
  $opts = [
    CURLOPT_URL => $job['url'],
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_FOLLOWLOCATION => true,
    CURLOPT_TIMEOUT => (int)$job['timeout_sec'],
    CURLOPT_HEADER => false,
    CURLOPT_HTTPHEADER => $hdrs
  ];
  if ($method !== 'GET') {
    $opts[CURLOPT_CUSTOMREQUEST] = $method;
    if (!empty($job['body'])) $opts[CURLOPT_POSTFIELDS] = $job['body'];
  }
  curl_setopt_array($ch, $opts);
  $start = microtime(true);
  $resp = curl_exec($ch);
  $err  = curl_error($ch) ?: null;
  $code = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
  curl_close($ch);
  $duration = (int)round((microtime(true)-$start)*1000);

  $ok = $err === null && $code >= 200 && $code < 400;
  $repo->insertLog([
    ':job_id'=>(int)$job['id'],
    ':started_at'=>gmdate('Y-m-d H:i:s'),
    ':finished_at'=>gmdate('Y-m-d H:i:s'),
    ':success'=>$ok ? 1 : 0,
    ':http_code'=>$code ?: null,
    ':duration_ms'=>$duration,
    ':response_preview'=>shorten($resp ?? '', 500),
    ':error'=>$err
  ]);
  $repo->touchLastRun((int)$job['id'], $ok ? 'OK' : 'FAIL', $code ?: null, $err);

  return [$ok, $code, $err];
}

if ($action === 'run' && isset($_GET['id'])) {
  $job = $repo->find((int)$_GET['id']);
  if ($job) { run_now($repo, $job, $config); }
  header('Location: ?a=list'); exit;
}

// HTML head
function head_html(string $title='WebCron PHP'){ ?>
<!doctype html>
<html lang="vi">
<head>
<meta charset="utf-8">
<meta name="viewport" content="width=device-width, initial-scale=1">
<title><?=h($title)?></title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/@tabler/core@1.0.0-beta20/dist/css/tabler.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.css" rel="stylesheet">
<style>
  body{ background:#0f172a; color:#e2e8f0 }
  .card{ background:#111827; border-color:#1f2937 }
  a, .nav-link{ color:#93c5fd }
  label{ color:#cbd5e1 }
  .form-control, .form-select, textarea{ background:#0b1220; color:#e5e7eb; border-color:#1f2937 }
  .badge-ok{ background:#16a34a } .badge-fail{ background:#dc2626 }
  code { color:#93c5fd }
</style>
</head>
<body>
<nav class="navbar navbar-dark" style="background:#111827">
  <div class="container">
    <a class="navbar-brand" href="?a=list"><i class="bi bi-clock-history me-2"></i>WebCron</a>
    <a class="btn btn-primary" href="?a=create"><i class="bi bi-plus-lg"></i> Thêm Job</a>
  </div>
</nav>
<div class="container my-4">
<?php }

function foot_html(){ ?>
</div>
</body>
</html>
<?php }

// Views
if ($action === 'list') {
  head_html('Jobs');
  $jobs = $repo->all(); ?>
  <div class="card shadow">
    <div class="card-body">
      <h3 class="mb-3">Danh sách Job</h3>
      <div class="table-responsive">
      <table class="table table-dark table-hover align-middle">
        <thead><tr>
          <th>ID</th><th>Tên</th><th>Schedule</th><th>URL</th><th>Bật</th>
          <th>Lần chạy gần nhất</th><th>Trạng thái</th><th>Tiếp theo</th><th></th>
        </tr></thead>
        <tbody>
        <?php foreach ($jobs as $j): ?>
          <tr>
            <td><?= (int)$j['id'] ?></td>
            <td><?= h($j['name']) ?></td>
            <td><code><?= h($j['cron_expr']) ?></code></td>
            <td class="text-truncate" style="max-width:260px"><?= h($j['url']) ?></td>
            <td><?= $j['enabled'] ? '✔' : '✖' ?></td>
            <td><?= h($j['last_run_at'] ?? '-') ?></td>
            <td>
              <?php if ($j['last_status'] === 'OK'): ?>
                <span class="badge badge-ok">OK</span>
              <?php elseif ($j['last_status'] === 'FAIL'): ?>
                <span class="badge badge-fail">FAIL</span>
              <?php else: ?>-<?php endif; ?>
            </td>
            <td>
              <?php
                try {
                  $next = (new Cron($j['cron_expr']))->getNextRunDate(new DateTimeImmutable('now'));
                  echo h($next->format('Y-m-d H:i'));
                } catch (Throwable $e) { echo '-'; }
              ?>
            </td>
            <td class="text-nowrap">
              <a class="btn btn-sm btn-outline-success" href="?a=run&id=<?=$j['id']?>"><i class="bi bi-play-fill"></i></a>
              <a class="btn btn-sm btn-outline-info" href="?a=logs&id=<?=$j['id']?>"><i class="bi bi-journal-text"></i></a>
              <a class="btn btn-sm btn-outline-warning" href="?a=edit&id=<?=$j['id']?>"><i class="bi bi-pencil"></i></a>
              <a class="btn btn-sm btn-outline-danger" href="?a=delete&id=<?=$j['id']?>" onclick="return confirm('Xóa job?')"><i class="bi bi-trash"></i></a>
            </td>
          </tr>
        <?php endforeach; ?>
        </tbody>
      </table>
      </div>
    </div>
  </div>
<?php foot_html(); exit; }

function form_job(?array $j=null){ ?>
  <div class="card shadow">
    <div class="card-body">
      <div class="row g-3">
        <div class="col-md-6">
          <label class="form-label">Tên</label>
          <input name="name" class="form-control" required value="<?=h($j['name']??'')?>">
        </div>
        <div class="col-md-3">
          <label class="form-label">Phương thức</label>
          <select name="method" class="form-select">
            <?php foreach (['GET','POST','PUT','PATCH','DELETE'] as $m): ?>
              <option <?=(($j['method']??'GET')===$m)?'selected':''?>><?=$m?></option>
            <?php endforeach; ?>
          </select>
        </div>
        <div class="col-md-3">
          <label class="form-label">Timeout (giây)</label>
          <input type="number" min="1" name="timeout_sec" class="form-control" value="<?=h((string)($j['timeout_sec']??30))?>">
        </div>
        <div class="col-12">
          <label class="form-label">URL</label>
          <input name="url" class="form-control" required value="<?=h($j['url']??'')?>">
        </div>
        <div class="col-12">
          <label class="form-label">Headers (JSON)</label>
          <textarea name="headers" rows="2" class="form-control" placeholder='{"Authorization":"Bearer ..."}'><?=h($j['headers']??'')?></textarea>
        </div>
        <div class="col-12">
          <label class="form-label">Body</label>
          <textarea name="body" rows="4" class="form-control"><?=h($j['body']??'')?></textarea>
        </div>
        <div class="col-md-6">
          <label class="form-label">Cron (* * * * *)</label>
          <input name="cron_expr" class="form-control" required value="<?=h($j['cron_expr']??'*/5 * * * *')?>">
        </div>
        <div class="col-md-6 d-flex align-items-end">
          <div class="form-check">
            <input class="form-check-input" type="checkbox" name="enabled" <?=(($j['enabled']??1)?'checked':'')?>>
            <label class="form-check-label">Bật job</label>
          </div>
        </div>

        <div class="col-12"><hr></div>

        <div class="col-12">
          <label class="form-label">Proxy (tùy chọn)</label>
          <input name="proxy_url" class="form-control" placeholder="http://host:port | socks5://host:1080" value="<?=h($j['proxy_url']??'')?>">
        </div>
        <div class="col-md-6">
          <label class="form-label">Proxy Auth (user:pass)</label>
          <input name="proxy_auth" class="form-control" value="<?=h($j['proxy_auth']??'')?>">
        </div>
        <div class="col-md-6 d-flex align-items-end">
          <div class="form-check">
            <input class="form-check-input" type="checkbox" name="verify_ssl" <?=(($j['verify_ssl']??1)?'checked':'')?>>
            <label class="form-check-label">Verify SSL (khuyến nghị bật)</label>
          </div>
        </div>
      </div>
      <div class="mt-3">
        <button class="btn btn-primary"><i class="bi bi-save"></i> Lưu</button>
        <a class="btn btn-secondary" href="?a=list">Hủy</a>
      </div>
    </div>
  </div>
<?php }

if ($action === 'create') {
  head_html('Tạo Job'); ?>
  <form method="post"><?php form_job(null); ?></form>
<?php foot_html(); exit; }

if ($action === 'edit' && isset($_GET['id'])) {
  $job = $repo->find((int)$_GET['id']); if (!$job) { header('Location:?a=list'); exit; }
  head_html('Sửa Job'); ?>
  <form method="post"><?php form_job($job); ?></form>
<?php foot_html(); exit; }

if ($action === 'logs' && isset($_GET['id'])) {
  $job = $repo->find((int)$_GET['id']); if (!$job) { header('Location:?a=list'); exit; }
  head_html('Logs - '.($job['name']??('Job #'.$job['id']))); ?>
  <div class="card shadow">
    <div class="card-body">
      <h4 class="mb-3">Logs của: <?=h($job['name'])?></h4>
      <?php $logs = $repo->logs((int)$job['id']); ?>
      <div class="table-responsive">
      <table class="table table-dark table-striped">
        <thead><tr><th>Thời gian</th><th>HTTP</th><th>OK?</th><th>ms</th><th>Preview</th><th>Lỗi</th></tr></thead>
        <tbody>
        <?php foreach ($logs as $l): ?>
          <tr>
            <td><?=h($l['started_at'])?></td>
            <td><?=h((string)$l['http_code'])?></td>
            <td><?= $l['success'] ? '✔' : '✖' ?></td>
            <td><?= (int)$l['duration_ms'] ?></td>
            <td><pre class="mb-0" style="white-space:pre-wrap;max-height:8rem;overflow:auto"><?=h($l['response_preview']??'')?></pre></td>
            <td><code><?=h($l['error']??'')?></code></td>
          </tr>
        <?php endforeach; ?>
        </tbody>
      </table>
      </div>
      <a class="btn btn-secondary" href="?a=list">Quay lại</a>
    </div>
  </div>
<?php foot_html(); exit; }

header('Location: ?a=list');
